#! /usr/bin/env julia
# ---------------------------------------------------------


# ---------------------------------------------------------
using CSV, FileIO
using DataFrames
using Parameters, Distributed
using Statistics: mean, std, cor, I, kron, UpperTriangular, median
using LaTeXStrings
using JLD2

push!(LOAD_PATH, "./src")
using project_routines
# ---------------------------------------------------------


# ---------------------------------------------------------
# logging
run(`mkdir -p log`)
using Logging, LoggingExtras, Dates

# clean up the files
map(x->rm(x, force=true), ["./log/table12_info.log.jl", "./log/table12_warn.log.jl", "./log/table12_full.log.jl"]);
# custom timestamps
const log_date_format = "yyyy-mm-dd HH:MM:SS"
timestamp_logger(logger) = TransformerLogger(logger) do log
  merge(log, (; message = "$(Dates.format(now(), log_date_format)) \n$(log.message)"))
end

# create the debugger
demux_logger = TeeLogger(
    MinLevelLogger(FileLogger("./log/table12_info.log.jl"), Logging.Info),
    MinLevelLogger(FileLogger("./log/table12_warn.log.jl"), Logging.Warn),
    MinLevelLogger(FileLogger("./log/table12_full.log.jl"), Logging.Debug),
    ConsoleLogger(stdout, Logging.Info),   # Common logger to be able to see info and warning in repl
) |> timestamp_logger |> global_logger;
# ---------------------------------------------------------


# ---------------------------------------------------------
## Load in the data
calibration_parameters = CSV.read("./input/param_calibration.csv", DataFrame);
Lj = CSV.read("./input/Ljstar.csv", DataFrame).Lj;
PjCj = CSV.read("./input/PjCj.csv", DataFrame);
Tau = CSV.read("./input/Taustar.csv", DataFrame) |> Matrix;

dict_in = load("./tmp/simulation_dict.jld2");
# ---------------------------------------------------------


# ---------------------------------------------------------
N = size(Tau, 1);
eta, psi, sigma, gamma, phi, years, dt, BURN_IN =  Vector(calibration_parameters[1, :]);

Pj_lambda = PjCj.Pj_lambda; 
Cj_lambda = PjCj.Cj_lambda;
lambdaj_pp_gamma0 = Pj_lambda ./ Cj_lambda .^ (-gamma);
lambdaj_pp_gamma = lambdaj_pp_gamma0 ./ sum(lambdaj_pp_gamma0);

countries_and_partners = [44, 7, [5 25 42], [23 24 35]];
calibration = parameters_TradeWar(eta, gamma, N, dt, phi,
    Lj, lambdaj_pp_gamma, Tau, 0.2, 1e-5);
# ---------------------------------------------------------


# ---------------------------------------------------------
dt_USACHN, dt_partners = TradeWar_USChina(
    countries_and_partners;
    calibration = calibration,
    Tau_inc_factor = 0.5,
    dict_out_org = dict_in)
# ---------------------------------------------------------


# ---------------------------------------------------------
CSV.write("./output/Table_12_A.csv", dt_USACHN)
CSV.write("./output/Table_12_B.csv", dt_partners)
# ---------------------------------------------------------


# ---------------------------------------------------------
t12a = CSV.read("./output/Table_12_A.csv", DataFrame)
t12b = CSV.read("./output/Table_12_B.csv", DataFrame)

table_header = LaTeXString(raw"""
% Table generated by table12.jl in julia ...

\begin{tabular}{lcccccc}
\toprule
""")


table_panelA = L"""
\multicolumn{7}{c}{Panel A: Effects on United States and China} \\
\midrule
      &       & \multicolumn{2}{c}{United States} &       & \multicolumn{2}{c}{China} \\
\cmidrule{3-4}\cmidrule{6-7}Variable &       & Before & After &       & Before & After \\
\midrule
Net Trade Cost, $\tau-1$ &     & %$(t12a[1,2]) & %$(t12a[1,3]) &       & %$(t12a[1,4]) & %$(t12a[1,5]) \\
Inward Resistance, $P$ &       & %$(t12a[2,2]) & %$(t12a[2,3]) &       & %$(t12a[2,4]) & %$(t12a[2,5]) \\
Outward Resistance, $\Pi$ &    & %$(t12a[3,2]) & %$(t12a[3,3]) &       & %$(t12a[3,4]) & %$(t12a[3,5]) \\
Import Share, $\omega$ &       & %$(t12a[4,2]) & %$(t12a[4,3]) &       & %$(t12a[4,4]) & %$(t12a[4,5]) \\
Unshared Risk, $\rho$ &        & %$(t12a[5,2]) & %$(t12a[5,3]) &       & %$(t12a[5,4]) & %$(t12a[5,5]) \\
\midrule
"""

table_panelB = L"""
\multicolumn{7}{c}{Panel B: Effects on Peripheral Countries, in Percent Change} \\
\midrule
      & Canada & Mexico & UK    & Korea & Malaysia & Singapore \\
\cmidrule{2-7}\emph{Trade Variables} &       &       &       &       &       &  \\
$P_j$    & %$(t12b[1,2]) & %$(t12b[1,3]) & %$(t12b[1,4])   & %$(t12b[1,5]) & %$(t12b[1,6]) & %$(t12b[1,7]) \\
$\Pi_j$  & %$(t12b[2,2]) & %$(t12b[2,3]) & %$(t12b[2,4])   & %$(t12b[2,5]) & %$(t12b[2,6]) & %$(t12b[2,7]) \\
$\omega_{USA,j}$ & %$(t12b[3,2]) & %$(t12b[3,3]) & %$(t12b[3,4])   & %$(t12b[3,5]) & %$(t12b[3,6]) & %$(t12b[3,7]) \\
$\omega_{CHN,j}$ & %$(t12b[4,2]) & %$(t12b[4,3]) & %$(t12b[4,4])   & %$(t12b[4,5]) & %$(t12b[4,6]) & %$(t12b[4,7]) \\
$\omega_{i,USA}$ & %$(t12b[5,2]) & %$(t12b[5,3]) & %$(t12b[5,4])   & %$(t12b[5,5]) & %$(t12b[5,6]) & %$(t12b[5,7]) \\
$\omega_{i,CHN}$ & %$(t12b[6,2]) & %$(t12b[6,3]) & %$(t12b[6,4])   & %$(t12b[6,5]) & %$(t12b[6,6]) & %$(t12b[6,7]) \\
      &       &       &       &       &       &  \\
\emph{Risk Measures} &       &       &       &       &       &  \\
$\overline{\sigma}_j$ & %$(t12b[7,2]) & %$(t12b[7,3]) & %$(t12b[7,4])   & %$(t12b[7,5]) & %$(t12b[7,6]) & %$(t12b[7,7]) \\
$R^2_{i,USA}$         & %$(t12b[8,2]) & %$(t12b[8,3]) & %$(t12b[8,4])   & %$(t12b[8,5]) & %$(t12b[8,6]) & %$(t12b[8,7]) \\
$R^2_{i,CHN}$         & %$(t12b[9,2]) & %$(t12b[9,3]) & %$(t12b[9,4])   & %$(t12b[9,5]) & %$(t12b[9,6]) & %$(t12b[9,7]) \\
$\rho_{i,USA}$        & %$(t12b[10,2]) & %$(t12b[10,3]) & %$(t12b[10,4])   & %$(t12b[10,5]) & %$(t12b[10,6]) & %$(t12b[10,7]) \\
$\rho_{i,CHN}$        & %$(t12b[11,2]) & %$(t12b[11,3]) & %$(t12b[11,4])   & %$(t12b[11,5]) & %$(t12b[11,6]) & %$(t12b[11,7]) \\
\bottomrule
\end{tabular}%
"""


# Export result to .tex file
write("./output/table_12.tex", 
  table_header * table_panelA * table_panelB);
# ---------------------------------------------------------





# ---------------------------------------------------------

